<?php

use CodeIgniter\Router\RouteCollection;

/**
 * @var RouteCollection $routes
 */

$routes->addRedirect('frontend/(.*)', '/');

// Rute publik untuk webhook dari payment gateway dan external link
$routes->post('webhook/payment', 'WebhookController::handlePayment');
$routes->get('verify/link/(:hash)', 'Frontend\AuthController::processVerificationLink/$1', ['as' => 'verify.link']);
$routes->get('profile/verify-email/(:hash)', 'Frontend\ProfileController::verifyNewEmail/$1', ['as' => 'profile.email.verify']);


// Rute ke login
$routes->get('/', 'Frontend\AuthController::showLoginForm', ['as' => 'login.show', 'filter' => 'guest']);

/**
 * Public Frontend Routes (No Login Required)
 */
$routes->group('', ['filter' => 'guest'], static function($routes) {
    $routes->get('login', 'Frontend\AuthController::showLoginForm');
    $routes->post('login', 'Frontend\AuthController::processLogin', ['as' => 'login.attempt']);
    $routes->get('register', 'Frontend\RegistrationController::showRegistrationForm', ['as' => 'register.show']);
    $routes->post('register', 'Frontend\RegistrationController::processRegistration', ['as' => 'register.create']);

    $routes->get('verify', 'Frontend\AuthController::showVerificationChoice', ['as' => 'verify.choice']);
    $routes->post('verify/send', 'Frontend\AuthController::sendVerificationLink', ['as' => 'verify.send']);
    
    // --- RUTE FORGOT PASSWORD ---
    $routes->get('forgot-password', 'Frontend\AuthController::showForgotPasswordForm', ['as' => 'forgot.form']);
    $routes->post('forgot-password', 'Frontend\AuthController::sendResetLink', ['as' => 'forgot.send']);
    $routes->get('reset-password/(:hash)', 'Frontend\AuthController::showResetForm/$1', ['as' => 'reset.form']);
    $routes->post('reset-password', 'Frontend\AuthController::processReset', ['as' => 'reset.process']);
});

/**
 * Protected Frontend Routes (Requires Login)
 */
$routes->group('', ['filter' => 'authUser'], static function($routes) {
    // --- Rute Dashboard & Logout ---
    $routes->get('dashboard', 'Frontend\DashboardController::showDashboard', ['as' => 'dashboard.show']);
    $routes->get('logout', 'Frontend\AuthController::logout', ['as' => 'logout']);

    // --- Rute Profil ---
    $routes->get('profile', 'Frontend\ProfileController::index', ['as' => 'profile.show']);
    $routes->post('profile/update-details', 'Frontend\ProfileController::updateDetails', ['as' => 'profile.update.details']);
    $routes->post('profile/update-password', 'Frontend\ProfileController::updatePassword', ['as' => 'profile.update.password']);
    $routes->post('profile/request-email-change', 'Frontend\ProfileController::requestEmailChange', ['as' => 'profile.email.request']);

    // --- Rute Transaksi ---
    $routes->get('transaction/new', 'Frontend\TransactionController::new', ['as' => 'transaction.new']);
    $routes->get('transaction/register/(:segment)', 'Frontend\TransactionController::showParticipantForm/$1', ['as' => 'transaction.participants.form']);
    $routes->post('transaction/create', 'Frontend\TransactionController::create', ['as' => 'transaction.create']);
    $routes->post('transaction/validate-promo', 'Frontend\TransactionController::validatePromo', ['as' => 'transaction.promo.validate']);
    $routes->get('transaction/(:segment)', 'Frontend\TransactionController::show/$1', ['as' => 'transaction.show']);
    $routes->get('transactions/pdf/(:segment)', 'Frontend\TransactionController::downloadPdf/$1', [
        'as' => 'transaction.pdf'
    ]);
    $routes->get('terms/(:segment)', 'Frontend\TransactionController::showTerms/$1', ['as' => 'terms.show']);

});


$routes->group('backend', static function($routes) {

    // --- Rute Login & Logout Admin ---
    $routes->get('login', 'Backend\AuthController::showLoginForm', ['as' => 'admin.login.show', 'filter' => 'guest:admin']);
    $routes->post('login', 'Backend\AuthController::processLogin', ['as' => 'admin.login.attempt']);
    $routes->get('logout', 'Backend\AuthController::logout', ['as' => 'admin.logout']);
    
    // Semua perlu login admin
    $routes->group('', ['filter' => 'authAdmin'], static function($routes) {

        // Util
        $routes->get('utils/clear-system-cache', 'Backend\DashboardController::clearCache', [
            'as' => 'admin.clear_cache',
            'filter' => 'permission:manage-system' // Anda perlu membuat permission ini
        ]);

        // Dashboard
        $routes->get('dashboard', 'Backend\DashboardController::index', ['as' => 'admin.dashboard']);

        // PROFILE
        $routes->get('profile', 'Backend\ProfileController::index', ['as' => 'admin.profile.show']);
        $routes->post('profile/update-password', 'Backend\ProfileController::updatePassword', ['as' => 'admin.profile.update_password']);

        // --- RUTE BARU UNTUK USER MANAGEMENT ---
        $routes->get('users', 'Backend\UsersController::index', ['as' => 'admin.users.index', 'filter' => 'permission:view-user-list']);
        $routes->get('users/edit/(:num)', 'Backend\UsersController::edit/$1', ['as' => 'admin.users.edit', 'filter' => 'permission:edit-user']);
        $routes->post('users/update/(:num)', 'Backend\UsersController::update/$1', ['as' => 'admin.users.update', 'filter' => 'permission:edit-user']);
        $routes->post('users/delete/(:num)', 'Backend\UsersController::delete/$1', ['as' => 'admin.users.delete', 'filter' => 'permission:delete-user']);
        
        // --- Rute Transaksi ---
        $routes->get('transactions', 'Backend\DataTransactionController::index', [
            'as' => 'admin.transactions.index',
            'filter' => 'permission:view-transaction-list'
        ]);
        $routes->get('transactions/show/(:segment)', 'Backend\DataTransactionController::show/$1', [
            'as' => 'admin.transactions.show',
            'filter' => 'permission:view-transaction-detail' // Gunakan permission baru
        ]);

        $routes->get('transactions/pdf/(:segment)', 'Backend\DataTransactionController::downloadPdf/$1', [
            'as' => 'admin.transactions.pdf', 
            'filter' => 'permission:view-transaction-detail'
        ]);

        $routes->resource('transactions', [
            'controller' => 'Backend\DataTransactionController',
            'as'         => 'admin.transactions',
            'placeholder'=> '(:segment)',
            'except'     => 'index, show' // Kecualikan index dan show
        ]);
        
        $routes->post('transactions/update-status/(:segment)', 'Backend\DataTransactionController::updateStatus/$1', [
            'as' => 'admin.transactions.update_status',
            'filter' => 'permission:edit-transaction'
        ]);

        $routes->post('transactions/resend-notification/(:segment)', 'Backend\DataTransactionController::resendNotification/$1', 
            ['as' => 'admin.transactions.resend_notification', 
            'filter' => 'permission:edit-transaction']);



        $routes->get('transactions/participants/edit/(:num)', 'Backend\DataTransactionController::editParticipant/$1', [
            'as' => 'admin.participants.edit', 
            'filter' => 'permission:edit-participant'
        ]);
        $routes->post('transactions/participants/update/(:num)', 'Backend\DataTransactionController::updateParticipant/$1', [
            'as' => 'admin.participants.update', 
            'filter' => 'permission:edit-participant'
        ]);

        // --- Rute Report Peserta ---
        $routes->get('reports/participants', 'Backend\ParticipantReportController::index', [
            'as' => 'admin.reports.participants',
            'filter' => 'permission:view-participant-report' // Kita butuh permission baru
        ]);

        $routes->get('reports/participants/pdf/(:num)', 'Backend\ParticipantReportController::downloadTicket/$1', [
            'as' => 'admin.reports.participant_pdf', 
            'filter' => 'permission:view-participant-report'
        ]);

        $routes->get('reports/participants/export', 'Backend\ParticipantReportController::exportExcel', [
            'as' => 'admin.reports.participants_export', 
            'filter' => 'permission:view-participant-report'
        ]);

        // --- Rute Admin ---
        $routes->get('admins', 'Backend\AdminsController::index', ['as' => 'admin.admins.index', 'filter' => 'permission:view-admin-list']);
        $routes->get('admins/new', 'Backend\AdminsController::new', ['as' => 'admin.admins.new', 'filter' => 'permission:create-admin']);
        $routes->post('admins', 'Backend\AdminsController::create', ['as' => 'admin.admins.create', 'filter' => 'permission:create-admin']);
        $routes->get('admins/edit/(:num)', 'Backend\AdminsController::edit/$1', ['as' => 'admin.admins.edit', 'filter' => 'permission:edit-admin']);
        $routes->post('admins/update/(:num)', 'Backend\AdminsController::update/$1', ['as' => 'admin.admins.update', 'filter' => 'permission:edit-admin']);
        $routes->post('admins/delete/(:num)', 'Backend\AdminsController::delete/$1', ['as' => 'admin.admins.delete', 'filter' => 'permission:delete-admin']);

        // --- Rute Roles ---
        $routes->get('roles', 'Backend\RolesController::index', ['as' => 'admin.roles.index', 'filter' => 'permission:view-role-list']);
        $routes->get('roles/new', 'Backend\RolesController::new', ['as' => 'admin.roles.new', 'filter' => 'permission:create-role']);
        $routes->post('roles', 'Backend\RolesController::create', ['as' => 'admin.roles.create', 'filter' => 'permission:create-role']);
        $routes->get('roles/edit/(:num)', 'Backend\RolesController::edit/$1', ['as' => 'admin.roles.edit', 'filter' => 'permission:edit-role']);
        $routes->post('roles/update/(:num)', 'Backend\RolesController::update/$1', ['as' => 'admin.roles.update', 'filter' => 'permission:edit-role']);

    });
});